/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef GUI_1CHANNEL_ODC_H
#define GUI_1CHANNEL_ODC_H

#include "config_handler.h"
#include "loggingclass.h"
#include "numpad.h"
#include "rf_system.h"
#include "rf_system_1channel_odc.h"
#include <QButtonGroup>
#include <QMainWindow>
#include <QMap>
#include <QPushButton>
#include <QTcpServer>
#include <QTcpSocket>
#include <QThread>
#include <QTimer>
#include <QtSerialPort/QSerialPort>

#ifdef ETHERCAT
#include "ethercat/module.h"
#endif

namespace Ui {
class GUI_1channel_ODC;
}

class GUI_1channel_ODC : public QMainWindow
{
	Q_OBJECT

public:
	explicit GUI_1channel_ODC(QWidget *parent = 0);
	~GUI_1channel_ODC();


signals:
	//MISCELLANEOUS
    void signal_temperature_sensor_init();
	void signal_channelInit(int intrasys_num);
	void signal_timerStart(int intrasys_num, int ms);
	void signal_timerStop(int intrasys_num);
	void signal_RF_enable(int intrasys_num, bool enable);
	void signal_setClock_source(int intrasys_num, int mode);		//0 = Standalone, 1 = Master, 2 = Slave (end), 3 = Slave (inline)
	void signal_set_power_control_mode(int intrasys_num, int mode);
	void signal_setExternal_triggering_enable(int intrasys_num, bool enable);
	void signal_execute_reset_SGx(int intrasys_num);
	void signal_execute_reset_protection(int intrasys_num);
	void signal_execute_error_clear(int intrasys_num);
	void signal_request_PA_power_update(int intrasys_num);
	void signal_set_modbus_enable(bool enable);
	void signal_set_ethercat_enable(bool enable);
	void signal_set_error_clearing_enable(int intrasys_num, bool enable);

	//CW
	void signal_setVGA_attenuation(int intrasys_num, double attenuation);
	void signal_setIQMod_magnitude(int intrasys_num, double magnitude);
	void signal_setPowerWatt(int intrasys_num, double power_watt);
	void signal_setPowerdBm(int intrasys_num, double power_dbm);
	void signal_setFrequency(int intrasys_num, double frequency);
	void signal_setPhase(int intrasys_num, double phase);

	//PWM
	void signal_setPWM_enable(int intrasys_num, bool enable);
	void signal_setPWM_frequency(int intrasys_num, double pwm_frequency);
	void signal_setPWM_duty_cycle(int intrasys_num, double pwm_duty_cycle);

	//DLL
	void signal_setDLL_enable(int intrasys_num, bool enable);
	void signal_setDLL_frequency_limit_upper(int intrasys_num, double all_frequency_limit_upper);
	void signal_setDLL_frequency_limit_lower(int intrasys_num, double all_frequency_limit_lower);
	void signal_setDLL_start_frequency(int intrasys_num, double start_frequency);
	void signal_setDLL_step_frequency(int intrasys_num, double step_frequency);
	void signal_setDLL_threshold (int intrasys_num, double dll_threshold);
	void signal_setDLL_main_delay(int intrasys_num, double main_delay);

	//ALL
	void signal_setALL_enable(int intrasys_num, bool enable);
	void signal_setALL_frequency_limit_upper(int intrasys_num, double all_frequency_limit_upper);
	void signal_setALL_frequency_limit_lower(int intrasys_num, double all_frequency_limit_lower);
	void signal_setALL_threshold(int intrasys_num, double all_threshold);

	//DVGA
	void signal_setDVGA_amplifier_forward_enable(int intrasys_num, bool enable);
	void signal_setDVGA_amplifier_reflected_enable(int intrasys_num, bool enable);
	void signal_setDVGA_attentuation_forward(int intrasys_num, double attenuation_fwd);
	void signal_setDVGA_attentuation_reflected(int intrasys_num, double attenuation_rfl);

	//PID
	void signal_setPID_kp(int intrasys_num, double value);
	void signal_setPID_ki(int intrasys_num, double value);
	void signal_setPID_kd(int intrasys_num, double value);
	void signal_setPID_setpoint(int intrasys_num, double value);
	void signal_setPID_scaling(int intrasys_num, double value);
	void signal_setPID_offset(int intrasys_num, double value);

	//SWEEP
	void signal_setSWP_power_dbm(int intrasys_num, double power_dbm);
	void signal_setSWP_power_watt(int intrasys_num, double power_watt);
	void signal_setSWP_start_frequency(int intrasys_num, double start_frequency);
	void signal_setSWP_stop_frequency(int intrasys_num, double stop_frequency);
	void signal_setSWP_step_frequency(int intrasys_num, double step_frequency);
	void signal_execute_SWP(int intrasys_num);

	//PSU
	void signal_setPSU_enable(int intrasys_num, bool enable);
	void signal_setPSU_voltage(int intrasys_num, double val);
	void signal_setPSU_current_limit(int intrasys_num, double val);

	//RCM
	void signal_RCM_Off();
	void signal_RCM_USB();
//	void signal_RCM_TCP();


/**********************************************************************************************************************************************************************************
 * SLOTS
 * *******************************************************************************************************************************************************************************/

public slots:
	void handler_execute_restart_program();

private slots:
#ifdef I2C_TEMP_SENSOR
	void handler_sensor_temperature_get(double val);
#endif

	//Miscellaneous Functions
	void handler_notification_get(int intrasys_num, QString notification);
	void handler_channel_working(int channel_number, bool enabled);
	void handler_channel_init_failed(int intrasys_num);
	void handler_firmware_version_get(int intrasys_num, int channel_num, QString version);
	void handler_RF_enable_get(int intrasys_num, bool RF_enabled);
	void handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection);
	void handler_error_get(int intrasys_num, quint64 error, QStringList error_messages);
	void handler_reset_detected(int intrasys_num, int channel_number);
	void handler_temperature_get(int intrasys_num, double temperature);
	void handler_clock_source_get(int intrasys_num, int mode);
	void handler_power_control_mode_get(int intrasys_num, int mode);
	void handler_modbus_connected_enable(bool enable, QString address, int port, QString error);
	void handler_interlock_enable_get(bool enable);

	//CW
	void handler_VGA_attenuation_get(int intrasys_num, double attenuation);
	void handler_IQMod_magnitude_get(int intrasys_num, double magnitude);
	void handler_power_get(int intrasys_num, double power_dbm, double power_watt);
	void handler_frequency_get(int intrasys_num, double frequency_mhz);
	void handler_phase_get(int intrasys_num, double phase_degrees);
	void handler_CW_enable_get(int intrasys_num, bool enable);

	//PWM
	void handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle);

	//DLL
	void handler_DLL_enable_get(int intrasys_num, bool enable);
	void handler_DLL_settings_get(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay);

	//ALL
	void handler_ALL_compatible_get(int intrasys_num, bool compatibility);
	void handler_ALL_enable_get(int intrasys_num, bool enable);
	void handler_ALL_settings_get(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage);

	//DVGA
	void handler_DVGA_compatible_get(int intrasys_num, bool compatibility);
	void handler_DVGA_forward_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);
	void handler_DVGA_reflected_settings_get(int intrasys_num, bool amplifier_enable, double attenuation);

	//PID
	void handler_PID_settings_get(int intrasys_num, double Kp, double Ki, double Kd, double setpoint, double scaling, double offset);

	//SWP
	void handler_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt);
//	void handler_SWP_measurement_get(int intrasys_num, QList<double> frequency, QList<double> power_FWD, QList<double> power_RFL);
	void handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data);

	//PSU
	void handler_PSU_voltage_setpoint_get(int intrasys_num, double val);
	void handler_PSU_current_limit_get(int intrasys_num, double val);
	void handler_PSU_enable_combined_get(int intrasys_num, bool enable);
	void handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power);
	void handler_PSU_power_efficiency_get(int intrasys_num, double efficiency);
	void handler_PSU_dissipation_get(int intrasys_num, double dissipation);

	//Data Logging
	void handler_datalogging_enable_get(bool enable);
	void handler_datalogging_storage_sufficient(bool valid);
	void handler_datalogging_log_deleted(bool success, QString filename);
    void handler_datalogging(bool enable);

	//Notification bar
	void show_notification(QString message, int time = 8000);
	void close_notification();
	void show_error(QString message);
	void close_error();

	//Miscellaneous buttons
	void on_pushButton_RF_enable_1_clicked();
	void on_pushButton_unit_power_reflected_1_clicked(bool checked);
	void on_pushButton_unit_power_forward_1_clicked(bool checked);
	void on_pushButton_unit_S11_1_clicked(bool checked);
	void on_pushButton_CW_enable_1_clicked();
	void on_pushButton_PWM_enable_1_clicked();
	void on_pushButton_ALL_enable_1_clicked();
	void on_pushButton_DLL_enable_1_clicked();

	//Numpad Buttons
	void on_numpadButton_0_clicked();
	void on_numpadButton_1_clicked();
	void on_numpadButton_2_clicked();
	void on_numpadButton_3_clicked();
	void on_numpadButton_4_clicked();
	void on_numpadButton_5_clicked();
	void on_numpadButton_6_clicked();
	void on_numpadButton_7_clicked();
	void on_numpadButton_8_clicked();
	void on_numpadButton_9_clicked();
	void on_numpadButton_period_clicked();
	void on_numpadButton_arrow_left_clicked();
	void on_numpadButton_arrow_right_clicked();
	void on_numpadButton_plus_clicked();
	void on_numpadButton_minus_clicked();
	void on_numpadButton_backspace_clicked();
	void on_numpadButton_clear_all_clicked();
	void on_numpadButton_ok_clicked();

	//Numpad Functions
	void numpad_value_confirmed_handler(QPushButton *button, double *num);

	//Menu Buttons
	void on_menu_home_Button_clicked();
	void on_menu_ALL_Button_clicked();
	void on_menu_DLL_Button_clicked();
	void on_menu_PID_Button_clicked();
	void on_menu_sweep_Button_clicked();
	void on_menu_settings_Button_clicked();
	void on_menu_help_Button_clicked();
	void on_menu_power_Button_clicked();
	void on_menu_PSU_Button_clicked();

	//Home Menu
	void on_pushButton_VGA_attenuation_1_clicked();
	void on_pushButton_IQMod_magnitude_1_clicked();
	void on_pushButton_power_dbm_1_clicked();
	void on_pushButton_power_watt_1_clicked();
	void on_pushButton_frequency_1_clicked();
	void on_pushButton_phase_1_clicked();

	void on_pushButton_PWM_frequency_1_clicked();
	void on_pushButton_PWM_duty_cycle_1_clicked();
	void on_pushButton_ALL_threshold_1_clicked();
	void on_pushButton_DLL_threshold_1_clicked();
	void on_pushButton_DLL_step_frequency_1_clicked();

	//ALL Menu
	void on_pushButton_ALL_frequency_limit_upper_1_clicked();
	void on_pushButton_ALL_frequency_limit_lower_1_clicked();
	void on_pushButton_ALL_threshold_2_clicked();

	void on_pushButton_ALL_reference_open_clicked();
	void on_pushButton_ALL_reference_back_clicked();	//back button the reference page

	void on_pushButton_DVGA_attenuation_forward_1_clicked();
	void on_pushButton_DVGA_attenuation_reflected_1_clicked();
	void on_pushButton_DVGA_amplifier_forward_1_clicked();
	void on_pushButton_DVGA_amplifier_reflected_1_clicked();

	//DLL Menu
	void on_pushButton_DLL_frequency_limit_upper_1_clicked();
	void on_pushButton_DLL_frequency_limit_lower_1_clicked();
	void on_pushButton_DLL_start_frequency_1_clicked();
	void on_pushButton_DLL_step_frequency_2_clicked();
	void on_pushButton_DLL_threshold_2_clicked();
	void on_pushButton_DLL_main_delay_1_clicked();
	void on_pushButton_power_dbm_2_clicked();
	void on_pushButton_power_watt_2_clicked();

	//PID Menu
	void on_pushButton_PID_kp_1_clicked();
	void on_pushButton_PID_ki_1_clicked();
	void on_pushButton_PID_kd_1_clicked();
	void on_pushButton_PID_setpoint_1_clicked();
	void on_pushButton_PID_scaling_1_clicked();
	void on_pushButton_PID_offset_1_clicked();

	//Sweep functions
	void on_pushButton_SWP_unit_clicked();
	void on_pushButton_SWP_execute_2_clicked();
	void on_pushButton_SWP_execute_1_clicked();
	void on_pushButton_SWP_back_clicked();

	void on_pushButton_SWP_start_frequency_1_clicked();
	void on_pushButton_SWP_stop_frequency_1_clicked();
	void on_pushButton_SWP_step_frequency_1_clicked();
	void on_pushButton_SWP_power_dbm_1_clicked();
	void on_pushButton_SWP_power_watt_1_clicked();

	//PSU Menu
	void on_pushButton_PSU_voltage_1_clicked();
	void on_pushButton_PSU_current_limit_1_clicked();
	void on_pushButton_PSU_enable_1_clicked(bool checked);

	//Settings Menu
	void on_pushButton_power_control_normal_1_clicked();			//Regular Power Control (auto-gain ON)
	void on_pushButton_power_control_feedforward_1_clicked();		//Feed Forward Power Control (auto-gain OFF)
	void on_pushButton_power_control_analog_1_clicked();			//Analog Input Power Control
	void on_pushButton_power_control_PID_1_clicked();				//PID Power Control

	void on_pushButton_data_logging_OFF_1_clicked();
	void on_pushButton_data_logging_ON_1_clicked();
//	void on_pushButton_data_logging_ON_1_toggled(bool checked);

	void on_pushButton_clock_source_standalone_1_clicked();		//Coherent mode: Off
	void on_pushButton_clock_source_master_1_clicked();			//Coherent mode: Master
	void on_pushButton_clock_source_slave_inline_1_clicked();	//Coherent mode: Slave (Inline)
	void on_pushButton_clock_source_slave_end_1_clicked();		//Coherent mode: Slave (End of line) //not used
	void on_pushButton_remote_command_OFF_1_clicked();			//RCM Off
	void on_pushButton_remote_command_USB_1_clicked();			//RCM USB
	void on_pushButton_remote_command_TCP_1_clicked();			//RCM TCP
	void on_pushButton_external_triggering_OFF_1_clicked();		//External Triggering OFF
	void on_pushButton_external_triggering_ON_1_clicked();		//External Triggering ON
	void on_pushButton_modbus_OFF_1_clicked();					//Modbus OFF
	void on_pushButton_modbus_TCP_slave_1_clicked();			//Modbus TCP Slave
	void on_pushButton_etherCAT_OFF_1_clicked();				//EtherCAT OFF
	void on_pushButton_etherCAT_ON_1_clicked();					//EtherCAT ON

    //Remote Command mode
	void UART_disconnected();
	void TCP_RCM_Pass();
	void TCP_RCM_Return();
	void TCP_newConnection();
	void TCP_disconnect_server();
	void TCP_client_disconnected();
	void RCM_controls_enable(bool state);

	//
	// TODO:
	// Fix these names!
	//
	//Overview Page buttons
	void on_pushButton_go_control_panel_1_clicked();

	//Power menu
	void on_pushButton_clear_errors_1_clicked();
	void on_pushButton_go_overview_1_clicked();
	void on_pushButton_reset_SGx_1_clicked();
	void on_pushButton_reset_protection_1_clicked();
	void on_pushButton_shutdown_controller_1_clicked();
	void on_pushButton_reboot_controller_1_clicked();
	void on_pushButton_restart_program_1_clicked();
	void on_pushButton_exit_program_1_clicked();
	void on_Exit_Yes_button_clicked();
	void on_Exit_No_button_clicked();


/**********************************************************************************************************************************************************************************
 * Variables and functions
 * *******************************************************************************************************************************************************************************/

public:
	RF_System_1channel_ODC *RF_system_constructor;
	ConfigHandler *config;
	LoggingClass *datalogger;
	
private:
	#ifdef ETHERCAT
	EtherCATModule ethercatModule;
	#endif
	
	Ui::GUI_1channel_ODC *ui;

	double numpad_value = 0;

	/* Remote Command Mode */
	void handler_RCM_mode();
	bool RCM_supported = false;
	int remote_command_mode = 0;	//0 = OFF, 1 = UART/USB, 2 = TCP
	QString ipAddress;				//IP Address of this system

	/* Modbus Mode */
	void handler_modbus_mode();
	
	/* EtherCAT Mode */
	void handler_ethercat_mode();

	/* SWEEP variables */
	void SWP_enable_GUI(bool enable);
	void SWP_draw_plot(QString SWP_raw_data);
	QString SWP_mode = "dbm";

	/* Coherency Mode */
	bool coherency_supported = false;

	/* CORE */
	QSerialPort SG_port;
	QSerialPort RCM_port;
	QTcpServer *tcpServer = nullptr;
	QTcpSocket *tcpSocket = nullptr;
	QString channel_select = "1";
	StatusCheck *status_checker;

	QTimer *notification_timer;

	bool RF_enabled = false;		//Actual RF power / ECS

	/* Startup functions */
	void firmware_version_check(QString firmware_version);
	int	firmware_version_requirement[4];
	QString	GUI_version;

	/* Miscellaneous */
	void configure_auto_error_clearing();

	void show_DLL_settings(bool input);
	void show_ALL_settings(bool input);
	void show_PWM_settings(bool input);
	void show_autogain_controls(int intrasys_num, bool enable);
	void show_GCS_controls(int intrasys_num, bool enable);
	void show_MCS_controls(int intrasys_num, bool enable);

	void set_power_reading_units_watt(bool mode);
	void menuButton_manager(int menuButton_ID);		//controls behaviour of menu buttons / page hopping

	QButtonGroup menu_buttonGroup;
	QButtonGroup clock_source_buttonGroup;
	QButtonGroup RCM_buttonGroup;
	QButtonGroup power_control_buttonGroup;
	QButtonGroup External_triggering_buttonGroup;
	QButtonGroup modbus_buttonGroup;
	QButtonGroup ethercat_buttonGroup;
	QButtonGroup logging_buttonGroup;

	int	last_page_index;		//Track the last menu visited for some miscellaneous use-cases
	int	power_menu_action;		//Choose Shutdown / Restart / Exit

	/* NUMPAD */
	/* target_parameter and target_button can be used to 'dynamically' refer to the most recently pressed 'control button' and accompanying relevant variable, on any of the pages.
	 * precision is used to decide how many decimals will be used.
	 * multiplier is for swapping between for example Hz and MHz.
	 * The contents of these four can be changed by using the numpad_preparation function below. */
	Numpad *numpad;
	QPushButton *target_parameter_button;
	double scale_multiplier;
	int precision;

	void numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision = 2, double scale = 1, bool period_enable = true, bool sign_enable = false);

	void ok_check_minmax(QPushButton *button);		//Check if the value fulfill the min/max requirements of an input.
};

#endif //GUI_1CHANNEL_ODC_H
